<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\RESTful\ResourceController;
use App\Models\AlimentModel;
use App\Models\BandeModel;
use App\Models\DistributionAlimentModel;
use CodeIgniter\API\ResponseTrait;

class AlimentController extends ResourceController
{
    use ResponseTrait;
    protected $format    = 'json';
    private $alimentModel;
    private $distributionModel;
    private $bande;
    private $validation;

    public function __construct()
    {
        $this->alimentModel = new AlimentModel();
        $this->bande = new BandeModel();
        $this->distributionModel = new DistributionAlimentModel();
        $this->validation = \Config\Services::validation();
    }

    /**
     * Return an array of resource objects, themselves in array format.
     *
     * @return ResponseInterface
     */
    public function index()
    {
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Aliments chargés avec succès',
            'message' => null,
            'data' => $this->alimentModel->orderBy('nom', 'ASC')
                                       ->findAll()
        ];
        return $this->respondCreated($response);
    }
    
    /**
     * Return the properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function show($id = null)
    {
        $data = $this->alimentModel->find($id);
        if (!$data) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aliment non trouvé',
                'message' => null,
                'data' => $data
            ];
        }
        else{
            $response = [
                'status' => 200,
                'error' => false,
                'messages' => 'Aliment chargé avec succès',
                'message' => null,
                'data' => $data
            ];
        }
        return $this->respondCreated($response);
    }

    /**
     * Return a new resource object, with default properties.
     *
     * @return ResponseInterface
     */
    public function new()
    {
        //
    }

    /**
     * Create a new resource object, from "posted" parameters.
     *
     * @return ResponseInterface
     */
    public function create() {
        $data = $this->request->getJSON(true);
        // Définir les règles de validation
        $rules = [
            'ferme_id'      => 'required|integer',
            'libelle'        => 'required|min_length[2]|max_length[100]|is_unique[aliment.nom]',
            'prix_unitaire'    => 'required|decimal',
            'typealiment'       => 'required',
            'nbre_sac'   => 'required|decimal',
            'quantite'    => 'required|decimal',
            'total'    => 'required|decimal',
            'fournisseur'        => 'required|min_length[2]|max_length[100]'
        ];
        $messages = [
            'ferme_id'      => ['required' => 'La bande est requise.'],
            'libelle'        => ['required' => 'Le nom de l\'aliment est requis.',
                                  'min_length' => 'Le nom de l\'aliment doit comporter au moins 2 caractères.',
                                  'max_length' => 'Le nom de l\'aliment ne doit pas dépasser 100 caractères.',
                                  'is_unique' => 'Un Enregistrement avec ce nom existe déjà.'],
            'fournisseur'        => ['required' => 'Le nom du fournisseur est requis.',
                                  'min_length' => 'Le nom du fournisseur doit comporter au moins 2 caractères.',
                                  'max_length' => 'Le du fournisseur ne doit pas dépasser 100 caractères.'],
            'prix_unitaire'   => ['required' => 'Le prix unitaire est requis.'],
            'typealiment'    => ['required' => 'Veillez choisir un type d\'aliment.'],
            'nbre_sac'    => ['integer' => 'Le nombre de sac doit être un nombre.'],
            'quantite'   => ['required' => 'La quantite est requis.'],
            'total'   => ['required' => 'Le Total est requis.']
        ];

        if (!$this->validate($rules, $messages)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de validation',
                'message' => $this->validator->getErrors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        $datas = [
            'ferme_id'         => $data['ferme_id'],
            'nom'=> $data['libelle'],
            'fournisseur' => $data['fournisseur'],
            'prix_unitaire'      => $data['prix_unitaire'],
            'type_aliment'      => $data['typealiment'],
            'nbre_sac'      => $data['nbre_sac'],
            'quantite'      => $data['quantite'],
            'total'      => $data['total']
        ];
        
        //recuperer les infos sur la bande active
        $bande = $this->bande->getActiveBandeByFerme($data['ferme_id']);
        if($bande){
            $datas['id_bande'] = $bande['id'];
        }

        if ($this->alimentModel->insert($datas) === false) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de l\'enregistrement de l\'aliment',
                'message' => $this->alimentModel->errors(),
                'data' => []
            ];
             return $this->respondCreated($response);
        }

        $datas['id'] = $this->alimentModel->getInsertID();
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Aliment Ajouté avec succès',
            'message' => '',
            'data' => $datas
        ];
        return $this->respondCreated($response);
    }

    /**
     * Return the editable properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function edit($id = null)
    {
        //
    }

    /**
     * Add or update a model resource, from "posted" properties.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function update($id = null)
    {
        $data = $this->request->getJSON(true);
        if (!$this->model->find($id)) {
            return $this->failNotFound('Aliment not found');
        }
        if ($this->model->update($id, $data) === false) {
            return $this->failValidationErrors($this->model->errors());
        }
        return $this->respond($this->model->find($id));
    }

    /**
     * Delete the designated resource object from the model.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function delete($id = null){
        if (!$this->alimentModel->find($id)) {
            return $this->failNotFound('Aliment not found');
        }
        $this->alimentModel->delete($id);
        return $this->respondDeleted(['id' => $id, 'message' => 'Aliment deleted']);
    }

    public function listeTypeAliment(){          
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Type Aliment Chargée Avec Succès',
            'message' => null,
            'data' => [
                ['id' => 'démarrage','text' => 'Démarrage'],
                ['id' => 'croissance','text' => 'Croissance'],
                ['id' => 'finition','text' => 'Finition']                
            ]
        ];
        return $this->respondCreated($response);
    }

    function listesAliments($idferme = null){
        if (is_null($idferme)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'ID de la ferme requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si la ferme existe
        if (!$this->alimentModel->getAlimentByFermeID($idferme,'active')) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucun aliment trouvé en stock pour cette ferme',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        $data = $this->alimentModel->getAlimentByFermeID($idferme,'active');
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Aliments chargés avec succès',
            'message' => null,
            'data' => $data
        ];
        return $this->respondCreated($response);
    }

    function stockOneAliment($idAliment = null){
        if (is_null($idAliment)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'ID de l\'aliment requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si l'aliment existe
        $data = $this->alimentModel->find($idAliment);
        if (!$data) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucun aliment trouvé en stock pour cette ferme',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        $stock = $this->distributionModel->getSumAlimentDistribuer($idAliment); 
        $reste = (int)$data['quantite'] - $stock;
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Aliments chargés avec succès',
            'message' => null,
            'data' => [
                'reste'=> $reste,
                'servie'=> $stock,
                'stockinitial'=> (int)$data['quantite'],
            ]
        ];
        return $this->respondCreated($response);
    }
    
    public function ListeAlimentUser($iduser = null){
        $data = $this->alimentModel->getAliments($iduser);
        
        if (empty($data)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucun aliment trouvé pour cet utilisateur',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
       
        foreach ($data as &$aliment) {
            $aliment['servie'] = (int)$this->distributionModel->getSumAlimentDistribuer($aliment['id']);
            // Calculer la quantité restante
            $aliment['reste'] = (int)$aliment['quantite']-(int)$aliment['servie'];
        }      

        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Aliments chargés avec succès',
            'message' => null,
            'data' => $data
        ];
        return $this->respondCreated($response);
    }
}
