<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\RESTful\ResourceController;
use App\Models\BandeModel;
use App\Models\MortaliteModel;
use App\Models\VenteModel;
use CodeIgniter\API\ResponseTrait;

class BandeController extends ResourceController
{
    use ResponseTrait;
    protected $format    = 'json';
    private $bandeModel;
    private $validation;
    private $mortalite;
    private $vente;

    public function __construct()
    {
        $this->bandeModel = new BandeModel();
        $this->mortalite = new MortaliteModel();
        $this->vente = new VenteModel();
        $this->validation = \Config\Services::validation();
    }

    /**
     * Return an array of resource objects, themselves in array format.
     *
     * @return ResponseInterface
     */
    public function index()
    {
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Bandes chargées avec succès',
            'message' => null,
            'data' => $this->bandeModel->loadBandeNot('delete')
        ];
        return $this->respondCreated($response);
    }

    /**
     * Return the properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function show($id = null)
    {
               
        $data = $this->bandeModel->getBandeById($id);
        if (!$data) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Bande non trouvée',
                'message' => null,
                'data' => $data
            ];
        }
        else{
            $response = [
                'status' => 200,
                'error' => false,
                'messages' => 'Bande chargée avec succès',
                'message' => null,
                'data' => $data
            ];
        }
        return $this->respondCreated($response);
    }

    /**
     * Create a new resource object, from "posted" parameters.
     *
     * @return ResponseInterface
     */
    public function create()
    {
        $data = $this->request->getJSON(true);

        
        // Définir les règles de validation
        $rules = [
            'ferme_id'      => 'required|integer',
            'libelle'        => 'required|min_length[2]|max_length[100]|is_unique[bande.libelle]',
            'souche'        => 'required|min_length[2]|max_length[100]',
            'entreprise'    => 'required|min_length[2]|max_length[100]',
            'nombreInitial'    => 'required|integer|min_length[1]',
            'nombreMort'       => 'permit_empty|integer',
            'poids_moyen'   => 'required|decimal',
            'date_debut'    => 'required|valid_date[Y-m-d]',
            'utilisateurId'      => 'required|integer',
            'prix_unitaire'   => 'required|decimal',
            'total'   => 'required|decimal',
        ];
        $messages = [
            'ferme_id'      => ['required' => 'La bande est requise.'],
            'libelle'        => ['required' => 'Le nom de la bande est requis.',
                                  'min_length' => 'Le nom de la bande doit comporter au moins 2 caractères.',
                                  'max_length' => 'Le nom de la bande ne doit pas dépasser 100 caractères.',
                                  'is_unique' => 'Une bande avec ce nom existe déjà.'],
            'souche'        => ['required' => 'La souche est requise.'],
            'entreprise'    => ['required' => "L'entreprise est requise."],
            'nombreInitial'    => ['required' => 'Le nombre initial est requis.'],
            'nombreMort'    => ['integer' => 'Le nombre de mois doit être un nombre.'],
            'poids_moyen'   => ['required' => 'Le poids moyen est requis.'],
            'date_debut'    => ['required' => 'La date de début est requise.'],
            'utilisateurId'    => ['required' => 'Veillez sélectionner un utilisateur.'],
            'prix_unitaire'   => ['required' => 'Le prix unitaire est requis.'],
            'total'   => ['required' => 'Le total est requis.']
        ];

        if (!$this->validate($rules, $messages)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de validation',
                'message' => $this->validator->getErrors(),
                'data' => [],
            ];
           
            return $this->respondCreated($response);
        }
        // Préparer les données à insérer
        $insertData = [
            'libelle'      => $data['libelle'],
            'ferme_id'      => $data['ferme_id'],
            'souche'        => $data['souche'],
            'entreprise'    => $data['entreprise'],
            'nb_initial'    => $data['nombreInitial'] ?? 0,
            'nb_mort'       => $data['nombreMort'] ?? 0,
            'poids_moyen'   => $data['poids_moyen'],
            'date_debut'    => $data['date_debut'],
            'statut'        => $data['statut'] ?? 'active',
            'prix_unitaire' => $data['prix_unitaire'] ?? 0,
            'total'         => $data['total'] ?? 0
        ];

        if ($this->bandeModel->insert($insertData) === false) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de la création de la bande',
                'message' => $this->validator->getErrors(),
                'data' => [],
            ];

            return $this->respondCreated($response);
        }

        $insertData['id'] = $this->bandeModel->getInsertID();
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Bande créée avec succès',
            'message' => null,
            'data' => $insertData,
        ];
        
        return $this->respondCreated($response);
    }

    /**
     * Add or update a model resource, from "posted" properties.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function update($id = null)
    {
        $data = $this->request->getJSON(true);

        // Définir les règles de validation
        $rules = [
            'id'      => 'required|integer',
            'libelle'       => 'required|min_length[2]|max_length[100]',
            'souche'        => 'required|min_length[2]|max_length[100]',
            'entreprise'    => 'required|min_length[2]|max_length[100]',
            'nombreInitial' => 'required|integer|min_length[1]',
            'nombreMort'    => 'permit_empty|integer',
            'poids_moyen'   => 'required|decimal',
            //'date_debut'    => 'required|valid_date[Y-m-d]',
            'date_fin'    => 'required|valid_date[Y-m-d]',
            'prix_unitaire' => 'required|decimal',
            'total'         => 'required|decimal',
        ];

        $messages = [
            'id'      => ['required' => 'La Bande est requise.'],
            'libelle'       => ['required' => 'Le nom de la bande est requis.',
                                          'min_length' => 'Le nom de la bande doit comporter au moins 2 caractères.',
                                          'max_length' => 'Le nom de la bande ne doit pas dépasser 100 caractères.',
                                          'is_unique' => 'Une bande avec ce nom existe déjà.'],
            'souche'        => ['required' => 'La souche est requise.'],
            'entreprise'    => ['required' => "L'entreprise est requise."],
            'nombreInitial'     => ['required' => 'Le nombre initial est requis.'],
            'nombreMort'        => ['integer' => 'Le nombre de mort doit être un nombre.'],
            'poids_moyen'    => ['required' => 'Le poids moyen est requis.'],
           // 'date_debut'     => ['required' => 'La date de début est requise.'],
            'date_fin'       => ['required' => 'La date de fin est requise.'],
            'prix_unitaire'  => ['required' => 'Le prix unitaire est requis.'],
            'total'          => ['required' => 'Le total est requis.']
        ];
        
        if (!$this->validate($rules, $messages)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de validation',
                'message' => $this->validator->getErrors(),
                'data' => [],
            ];
            return $this->respondCreated($response);
        }

        // Préparer les données à mettre à jour
        $updateData = [
            'libelle'      => $data['libelle'],
            'souche'        => $data['souche'],
            'entreprise'    => $data['entreprise'],
            'nb_initial'    => $data['nombreInitial'] ?? 0,
            'nb_mort'       => $data['nombreMort'] ?? 0,
            'poids_moyen'   => $data['poids_moyen'] ?? 0,
            //'date_debut'    => $data['date_debut'],
            'date_fin'      => $data['date_fin'],
            'statut'        => $data['statut'] ?? 'active',
            'prix_unitaire' => $data['prix_unitaire'] ?? 0,
            'total'         => $data['total'] ?? 0
        ];

        if ($this->bandeModel->update($id, $updateData) === false) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de la modification de la bande',
                'message' => $this->bandeModel->errors(),
                'data' => [],
            ];
            return $this->respondCreated($response);
        }

        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Bande modifiée avec succès',
            'message' => null,
            'data' => $this->bandeModel->find($id),
        ];
        return $this->respondCreated($response);
    }

    /**
     * Delete the designated resource object from the model.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function delete($id = null) {
        // Vérifier si l'ID de la bande est fourni
        if (is_null($id)) {
            $response = [
                'status' => 200,
                'error' => true,    
                'messages' => 'ID de la bande requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si la bande existe 
        if (!$this->bandeModel->find($id)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Bande non trouvée',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        $datas = [
            'statut'      => 'terminer', // Valeur par défaut si non fournie 
            'date_fin' => date('Y-m-d H:i:s'), // Date de fin actuelle
        ];

        // Supprimer la bande       
        if ($this->bandeModel->update($id,$datas) === false) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de la modification de la bande',
                'message' => $this->bandeModel->errors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        else{
             $response = [
                'status' => 200,
                'error' => false,
                'messages' => 'Bande Terminée avec succès',
                'message' => '',
                'data' => $datas
            ];
            return $this->respondCreated($response);
        }
    }

    public function loadBandeByFerme($idFerme = null, $userId = null) {
        if (is_null($idFerme)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'ID de la ferme requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si la ferme existe
        if (!$this->bandeModel->getBandeByFerme($idFerme, $userId)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucune bande trouvée pour cette ferme',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        $data = $this->bandeModel->getBandeByFerme($idFerme, $userId);
         //recuperer les infos sur la bande active
        $bande = $this->bandeModel->getActiveBandeByFerme($userId);
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Bandes chargées avec succès',
            'message' => null,
            'data' => $data,
            'resteSujet' => $this->mortalite->getSujetRestant($bande['id'])-$this->vente->getQuantiteVente($bande['id'])
        ];
        return $this->respondCreated($response);
    }
    
}