<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\RESTful\ResourceController;
use App\Models\BeneficeVenteBandeModel;
use App\Models\CoutProductionBandeModel;
use CodeIgniter\API\ResponseTrait;

class BeneficeVenteBandeController extends ResourceController
{
    use ResponseTrait;
    protected $format = 'json';
    private $beneficeModel;
    private $coutModel;

    public function __construct()
    {
        $this->beneficeModel = new BeneficeVenteBandeModel();
        $this->coutModel = new CoutProductionBandeModel();
    }

    public function index()
    {
        $data = $this->beneficeModel->orderBy('date_vente', 'DESC')->findAll();
        return $this->respond(['status' => 200, 'error' => false, 'message' => null, 'data' => $data]);
    }

    public function show($id = null)
    {
        $data = $this->beneficeModel->find($id);
        if (!$data) return $this->failNotFound('Benefice not found');
        return $this->respond($data);
    }

    public function showByVente($vente_id = null)
    {
        $data = $this->beneficeModel->where('vente_id', $vente_id)->first();
        if (!$data) return $this->failNotFound('Benefice not found for this vente');
        return $this->respond($data);
    }

    public function create()
    {
        $data = $this->request->getJSON(true);
        $rules = [
            'vente_id' => 'required|integer',
            'bande_id' => 'required|integer',
            'date_vente' => 'required|valid_date[Y-m-d]',
            'montant_vente' => 'required|decimal',
            'quantite' => 'required|integer',
            'prix_unitaire' => 'required|decimal'
        ];
        if (!$this->validate($rules)) {
            return $this->respondCreated(['status' => 200, 'error' => true, 'messages' => 'Validation error', 'message' => $this->validator->getErrors(), 'data' => []]);
        }

        // avoid duplicate for same vente
        if ($this->beneficeModel->where('vente_id', $data['vente_id'])->first()) {
            return $this->respondCreated(['status' => 200, 'error' => true, 'messages' => 'Benefice already exists for this vente', 'data' => []]);
        }

        // Find cost for the date, fallback to latest before
        $coutRow = $this->coutModel->getCostByDate($data['bande_id'], $data['date_vente']);
        if (!$coutRow) {
            $coutRow = $this->coutModel->getLatestCostBefore($data['bande_id'], $data['date_vente']);
        }
        $cout_val = $coutRow ? (float)$coutRow['cout_production'] : 0.00;
        $quantite = isset($data['quantite']) ? (int)$data['quantite'] : 1;
        $cout_total = $cout_val * $quantite;
        $benefice = (float)$data['montant_vente'] - $cout_total;
        $prix_unitaire = $quantite > 0 ? $cout_total / $quantite : 0;
        $benefice_unitaire = $data['prixUnitaire'] - $cout_val;

        $insert = [
            'vente_id' => $data['vente_id'],
            'bande_id' => $data['bande_id'],
            'ferme_id' => $data['ferme_id'],
            'date_vente' => $data['date_vente'],
            'montant_vente' => $data['montant_vente'],
            'quantite' => $quantite,
            'cout_production' => $cout_val,
            'benefice' => $benefice,
            'note' => $data['note'] ?? null,
            'prix_unitaire' => $prix_unitaire,
            'benefice_unitaire' => $benefice_unitaire
        ];

        $id = $this->beneficeModel->insert($insert);
        if (!$id) {
            return $this->respondCreated(['status' => 200, 'error' => true, 'messages' => 'Erreur lors de l\'enregistrement', 'message' => $this->beneficeModel->errors(), 'data' => []]);
        }
        $insert['id'] = $id;
        return $this->respondCreated(['status' => 201, 'error' => false, 'messages' => 'Enregistré', 'data' => $insert]);
    }

    public function update($id = null)
    {
        $payload = $this->request->getJSON(true);
        if (!$this->beneficeModel->find($id)) return $this->failNotFound('Benefice not found');
        if ($this->beneficeModel->update($id, $payload) === false) {
            return $this->failValidationErrors($this->beneficeModel->errors());
        }
        return $this->respond($this->beneficeModel->find($id));
    }

    public function delete($id = null)
    {
        if (!$this->beneficeModel->find($id)) return $this->failNotFound('Benefice not found');
        $this->beneficeModel->delete($id);
        return $this->respondDeleted(['id' => $id, 'message' => 'Benefice deleted']);
    }

    //calculer le total des benefices pour une ferme et une bande
    public function calculerTotalBenefices($ferme_id = null, $bande_id = null)
    {
        $builder = $this->beneficeModel->builder();
        // alias the sum column for predictable key name
        $builder->select('sum(benefice) as total_benefice');

        if ($ferme_id !== null) {
            $builder->where('ferme_id', $ferme_id);
        }
        if ($bande_id !== null) {
            $builder->where('bande_id', $bande_id);
        }

        $row = $builder->get()->getRowArray();

        $total = 0.00;
        if ($row && array_key_exists('total_benefice', $row) && $row['total_benefice'] !== null) {
            $total = (float) $row['total_benefice'];
        }

        $response = [
            'status' => 200,
            'error' => false,
            'message' => null,
            'data' => ['total_benefice' => $total]
        ];

        return $this->respond($response);
    }

    //calculer le total des ventes pour une ferme et une bande
  /*
    public function calculerTotalVentes($ferme_id = null, $bande_id = null)
    {
        $builder = $this->beneficeModel->builder();
        $builder->selectSum('montant_vente');
        if ($ferme_id !== null) {
            $builder->where('ferme_id', $ferme_id);
        }
        if ($bande_id !== null) {
            $builder->where('bande_id', $bande_id);
        }
        $query = $builder->get();
        $result = $query->getRowArray();
        if($result === null) {
            return $this->respondCreated([
                'status' => 200, 
                'error' => false, 
                'message' => null, 
                'data' => ['total_vente' => 0.00]]);
        }
        $total_vente = $result['montant_vente'] ?? 0.00;
        return $this->respondCreated([
            'status' => 200, 
            'error' => false, 
            'message' => null, 
            'data' => ['total_vente' => (float)$total_vente]]);
    }
            */
}