<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\RESTful\ResourceController;
use CodeIgniter\API\ResponseTrait;
use App\Models\AlimentModel;
use App\Models\DistributionAlimentModel;
use App\Models\FermeModel;
use App\Models\BandeModel;

class DistributionAlimentController extends ResourceController
{
    use ResponseTrait;
    protected $format    = 'json';
    private $alimentModel;
    private $distributionModel;
    private $fermeModel;
    private $validation;
    private $bande;

    public function __construct()
    {
        $this->alimentModel = new AlimentModel();
        $this->distributionModel = new DistributionAlimentModel();
        $this->fermeModel = new FermeModel();
        $this->bande = new BandeModel();
        $this->validation = \Config\Services::validation();
    }

    /**
     * Return an array of resource objects, themselves in array format.
     *
     * @return ResponseInterface
     */
    public function index()
    {
        return $this->respond($this->model->findAll());
    }

    /**
     * Return the properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function show($id = null)
    {
        $data = $this->model->find($id);
        if (!$data) {
            return $this->failNotFound('DistributionAliment not found');
        }
        return $this->respond($data);
    }

    /**
     * Create a new resource object, from "posted" parameters.
     *
     * @return ResponseInterface
     */
    public function create()
    {
        $data = $this->request->getJSON(true);
        // Définir les règles de validation
        $rules = [
            'ferme_id'      => 'required|integer',
            'aliment_id'        => 'required|integer',
            'bande_id'    => 'required|integer',
            'date_distribution'       => 'required|valid_date[Y-m-d]',
            'quantite'    => 'required|decimal',
        ];

        $messages = [
            'ferme_id'      => ['required' => 'La ferme est requise.'],
            'bande_id'      => ['required' => 'La bande est requise.'],
            'aliment_id'      => ['required' => 'L\'aliment est requis.'],
            'quantite'   => ['required' => 'La quantite est requise.'],
            'date_distribution'  => ['required' => 'La date de distribution est requise.'],
        ];

        if (!$this->validate($rules, $messages)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de validation',
                'message' => $this->validator->getErrors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        $datas = [
            'bande_id'         => $data['bande_id'],
            'aliment_id'         => $data['aliment_id'],
            'date'      => $data['date_distribution'],
            'quantite'      => $data['quantite']
        ];

        if ($this->distributionModel->insert($datas) === false) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de l\'enregistrement!!!',
                'message' => $this->distributionModel->errors(),
                'data' => []
            ];
             return $this->respondCreated($response);
        }

        $datas['id'] = $this->distributionModel->getInsertID();
        //verifier si le stock d'aliment est à zero et mettre le statut de ce stock à terminer
        $table_aliment = $this->alimentModel->find($data['aliment_id']); //stock initial
        $stock = $this->distributionModel->getSumAlimentDistribuer($data['aliment_id']); //quantité déjà distribuée

        if((int)$table_aliment['quantite'] - (int)$stock ==0){
            $this->alimentModel->update($data['aliment_id'], ['statut'=>'terminer']);
        }

        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Aliment Ajouté avec succès',
            'message' => null,
            'acheter' => (int)$table_aliment['quantite'],
            'distribuer' => (int)$stock,
            'data' => $datas
        ];
        return $this->respondCreated($response);
    }

    /**
     * Add or update a model resource, from "posted" properties.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function update($id = null)
    {
        $data = $this->request->getJSON(true);
        if (!$this->model->find($id)) {
            return $this->failNotFound('DistributionAliment not found');
        }
        if ($this->model->update($id, $data) === false) {
            return $this->failValidationErrors($this->model->errors());
        }
        return $this->respond($this->model->find($id));
    }

    /**
     * Delete the designated resource object from the model.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function delete($id = null)
    {
        if (!$this->model->find($id)) {
            return $this->failNotFound('DistributionAliment not found');
        }
        $this->model->delete($id);
        return $this->respondDeleted(['id' => $id, 'message' => 'DistributionAliment deleted']);
    }

    function listesDistributionsAliments($idUser = null){
        if (is_null($idUser)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'ID de la ferme requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si la ferme existe
        // Charger les fermes de l'utilisateur
        if (!$this->fermeModel->loadFermeByUserId($idUser,'active')) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucune ferme trouvés pour cet utilisateur',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        //recuperer les infos sur la bande active
        $bandeInfos = $this->bande->getActiveBandeByFerme($idUser); // $bande['id']

        $data = $this->distributionModel->getAlimentDistribuerByUSerID($idUser, $bandeInfos['id']);
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Aliments chargés avec succès',
            'message' => null,
            'data' => $data
        ];
        return $this->respondCreated($response);
    }
    
}
