<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\RESTful\ResourceController;
use CodeIgniter\API\ResponseTrait;
use App\Models\MedicamentModel;
use App\Models\FermeModel;
use App\Models\DistributionMedicamentModel;
use App\Models\BandeModel;

class DistributionMedicamentController extends ResourceController
{
    use ResponseTrait;
    protected $format    = 'json';
    private $medicamentModel;
    private $distributionModel;
    private $fermeModel;
    private $bandeModel;
    private $validation;

    public function __construct()
    {
        $this->medicamentModel = new MedicamentModel();
        $this->distributionModel = new DistributionMedicamentModel();
        $this->fermeModel = new FermeModel();
        $this->bandeModel = new BandeModel();
        $this->validation = \Config\Services::validation();
    }

    /**
     * Return an array of resource objects, themselves in array format.
     *
     * @return ResponseInterface
     */
    public function index()
    {
        //
    }

    /**
     * Return the properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function show($id = null)
    {
        //
    }

    /**
     * Return a new resource object, with default properties.
     *
     * @return ResponseInterface
     */
    public function new()
    {
        //
    }

    /**
     * Create a new resource object, from "posted" parameters.
     *
     * @return ResponseInterface
     */
    public function create(){
        $data = $this->request->getJSON(true);
        // Définir les règles de validation
        $rules = [
            'ferme_id'      => 'required|integer',
            'medicament_id'        => 'required|integer',
            'bande_id'    => 'required|integer',
            'date_distribution'       => 'required|valid_date[Y-m-d]',
            'quantite'    => 'required|decimal',
        ];

        $messages = [
            'ferme_id'      => ['required' => 'La ferme est requise.'],
            'bande_id'      => ['required' => 'La bande est requise.'],
            'medicament_id	'      => ['required' => 'Le médicament est requis.'],
            'quantite'   => ['required' => 'La quantite est requise.'],
            'date_distribution'  => ['required' => 'La date de distribution est requise.'],
        ];

        if (!$this->validate($rules, $messages)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de validation',
                'message' => $this->validator->getErrors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        $datas = [
            'bande_id'         => $data['bande_id'],
            'medicament_id'         => $data['medicament_id'],
            'date'      => $data['date_distribution'],
            'dose'      => $data['quantite']
        ];

        if ($this->distributionModel->insert($datas) === false) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de l\'enregistrement!!!',
                'message' => $this->distributionModel->errors(),
                'data' => []
            ];
             return $this->respondCreated($response);
        }

        $datas['id'] = $this->distributionModel->getInsertID();
        //verifier si le stock de medicament est à zero et mettre le statut de ce stock à terminer
        $table_medicament = $this->medicamentModel->find($data['medicament_id']); //stock initial
        $stock = $this->distributionModel->getSumMedicamentDistribuer($data['medicament_id']); //quantité déjà distribuée

        if((int)$table_medicament['quantite'] - (int)$stock ==0){
            $this->medicamentModel->update($data['medicament_id'], ['statut'=>'terminer']);
        }

        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Médicament Ajouté avec succès',
            'message' => null,
            'acheter' => (int)$table_medicament['quantite'],
            'distribuer' => (int)$stock,
            'data' => $datas
        ];
        return $this->respondCreated($response);
    }

    /**
     * Return the editable properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function edit($id = null)
    {
        //
    }

    /**
     * Add or update a model resource, from "posted" properties.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function update($id = null)
    {
        //
    }

    /**
     * Delete the designated resource object from the model.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function delete($id = null)
    {
        //
    }

    function listesDistributionsMedicaments($idUser = null){
        if (is_null($idUser)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'ID de la ferme requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si la ferme existe
        // Charger les fermes de l'utilisateur
        if (!$this->fermeModel->loadFermeByUserId($idUser,'active')) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucune ferme trouvés pour cet utilisateur',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        
        //recuperer les infos sur la bande active
        $bandeInfos = $this->bandeModel->getActiveBandeByFerme($idUser);
        $data = $this->distributionModel->getMedicamentDistribuerByUSerID($idUser, $bandeInfos['id']);
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Médicaments chargés avec succès',
            'message' => null,
            'data' => $data
        ];
        return $this->respondCreated($response);
    }

    function stockOneMedicament($idMedicament = null){
        if (is_null($idMedicament)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'ID du médicament requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si le médicament existe
        $data = $this->medicamentModel->find($idMedicament);
        if (!$data) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucun médicament trouvé en stock pour cette ferme',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        $stock = $this->distributionModel->getSumMedicamentDistribuer($idMedicament,2); //quantité déjà distribuée
        $reste = (int)$data['quantite'] - $stock;
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Médicaments chargés avec succès',
            'message' => null,
            'data' => [
                'reste'=> $reste,
                'servie'=> $stock,
                'stockinitial'=> (int)$data['quantite'],
            ]
        ];
        return $this->respondCreated($response);
    }
}
