<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\RESTful\ResourceController;
use App\Models\FermeModel;
use CodeIgniter\API\ResponseTrait;

class FermeController extends ResourceController
{
    use ResponseTrait;
    protected $format    = 'json';
    private $fermeModel;
    private $validation;

    public function __construct()
    {
        $this->fermeModel = new FermeModel();
        $this->validation = \Config\Services::validation();
    }

    /**
     * Return an array of resource objects, themselves in array format.
     *
     * @return ResponseInterface
     */
    public function index()
    {
        //return $this->respond($this->fermeModel->findAll());
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Fermes chargées avec succès',
            'message' => null,
            'data' => $this->fermeModel->loadFerme('active')
           /* ->where('statut', 'active')
                                     ->orderBy('nom', 'ASC')
                                       ->findAll() */
        ];
        return $this->respondCreated($response);
    }

    /**
     * Return the properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function show($id = null)
    {
               
        $data = $this->fermeModel->find($id);
        if (!$data) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Ferme non trouvée',
                'message' => null,
                'data' => $data
            ];
        }
        else{
            $response = [
                'status' => 200,
                'error' => false,
                'messages' => 'Ferme chargée avec succès',
                'message' => null,
                'data' => $data
            ];
        }
        return $this->respondCreated($response);
    }

    /**
     * Return a new resource object, with default properties.
     *
     * @return ResponseInterface
     */
    public function new()
    {
        //
    }

    /**
     * Create a new resource object, from "posted" parameters.
     *
     * @return ResponseInterface
     */
    public function create()
    {
        $data = $this->request->getJSON(true);

        // Définir les règles de validation
        $rules = [
            'nom_ferme'         => 'required|min_length[5]|max_length[100]|is_unique[ferme.nom]',
            'localisation_ferme'=> 'required|min_length[5]|max_length[100]',
        ];
        $messages = [
            'nom_ferme' => [
                'required'   => 'Le nom de la ferme est requis.',
                'min_length' => 'Le nom doit contenir au moins 5 caractères.',
                'max_length' => 'Le nom doit contenir au plus 100 caractères.',
                'is_unique'  => 'Le nom de la ferme doit être unique.'
            ],
            'localisation_ferme' => [
                'required'   => 'La localisation est requise.',
                'min_length' => 'La localisation doit contenir au moins 5 caractères.',
                'max_length' => 'La localisation doit contenir au plus 100 caractères.'
            ]
        ];

        if (!$this->validate($rules, $messages)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de validation',
                'message' => $this->validator->getErrors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        $datas = [
            'nom'         => $data['nom_ferme'],
            'localisation'=> $data['localisation_ferme'],
            'utilisateur_id' => $data['utilisateurId'], // Assurez-vous que l'ID de l'utilisateur est fourni
            'statut'      => 'active', // Valeur par défaut si non fournie 
            // Ajoute d'autres champs si besoin
        ];

        if ($this->fermeModel->insert($datas) === false) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de la création de la ferme',
                'message' => $this->fermeModel->errors(),
                'data' => []
            ];
             return $this->respondCreated($response);
        }

        $datas['id'] = $this->fermeModel->getInsertID();
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Ferme créée avec succès',
            'message' => '',
            'data' => $datas
        ];
        return $this->respondCreated($response);
    }

    /**
     * Return the editable properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function edit($id = null)
    {
        //
    }

    /**
     * Add or update a model resource, from "posted" properties.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function update($id = null) {
        $data = $this->request->getJSON(true);

        // Définir les règles de validation
        $rules = [
            'nom_ferme'         => 'required|min_length[5]|max_length[100]|is_unique[ferme.nom]',
            'localisation_ferme'=> 'required|min_length[5]|max_length[100]',
            'ferme_id'          => 'required|is_not_unique[ferme.id,id,{id}]', // Assurez-vous que l'ID de la ferme est fourni et valide
        ];
        $messages = [
            'nom_ferme' => [
                'required'   => 'Le nom de la ferme est requis.',
                'min_length' => 'Le nom doit contenir au moins 5 caractères.',
                'max_length' => 'Le nom doit contenir au plus 100 caractères.',
                'is_unique'  => 'Le nom de la ferme doit être unique.'
            ],
            'localisation_ferme' => [
                'required'   => 'La localisation est requise.',
                'min_length' => 'La localisation doit contenir au moins 5 caractères.',
                'max_length' => 'La localisation doit contenir au plus 100 caractères.'
            ],
            'ferme_id' => [
                'required' => 'L\'ID de la ferme est requis.',
                'is_not_unique' => 'L\'ID de la ferme doit être valide.'
            ]
        ];

        if (!$this->validate($rules, $messages)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de validation',
                'message' => $this->validator->getErrors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        $datas = [
            //'id'          => $data['ferme_id'], // Assurez-vous que l'ID de la ferme est fourni
            'nom'         => $data['nom_ferme'],
            'localisation'=> $data['localisation_ferme'],
            //'statut'      => 'active', // Valeur par défaut si non fournie 
        ];
        // Vérifier si la ferme existe
        if (!$this->fermeModel->find($data['ferme_id'])) {
            $response = [
                'status' => 200,        
                'error' => true,
                'messages' => 'Ferme non trouvée',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Mettre à jour la ferme        
        if ($this->fermeModel->update($data['ferme_id'],$datas) === false) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de la Modification de la ferme',
                'message' => $this->fermeModel->errors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        else{
             $response = [
                'status' => 200,
                'error' => false,
                'messages' => 'Ferme Modifiée avec succès',
                'message' => '',
                'data' => $datas
            ];
            return $this->respondCreated($response);
        }
    }

    /**
     * Delete the designated resource object from the model.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function delete($id = null) {
        // Vérifier si l'ID de la ferme est fourni
        if (is_null($id)) {
            $response = [
                'status' => 200,
                'error' => true,    
                'messages' => 'ID de la ferme requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si la ferme existe 
        if (!$this->fermeModel->find($id)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Ferme non trouvée',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        $datas = [
            'statut'      => 'delete', // Valeur par défaut si non fournie 
        ];

        // Supprimer la ferme       
        if ($this->fermeModel->update($id,$datas) === false) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de la suppression de la ferme',
                'message' => $this->fermeModel->errors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        else{
             $response = [
                'status' => 200,
                'error' => false,
                'messages' => 'Ferme supprimée avec succès',
                'message' => '',
                'data' => $datas
            ];
            return $this->respondCreated($response);
        }
    }   

    public function listUserFerme($userId = null)
    {
        // Vérifier si l'ID de l'utilisateur est fourni
        if (is_null($userId)) {
            $response = [
                'status' => 200,
                'error' => true,    
                'messages' => 'ID de l\'utilisateur requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        // Charger les fermes de l'utilisateur
        $data = $this->fermeModel->loadFermeByUserId($userId,'active');
                        
        // Vérifier si des fermes existent pour cet utilisateur
        if(!$data)  {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucune ferme trouvée pour cet utilisateur',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        
        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Fermes chargées avec succès',
            'message' => null,
            'data' => $data
        ];
        return $this->respondCreated($response);
    }
}
