<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\RESTful\ResourceController;
use App\Models\DistributionMedicamentModel;
use App\Models\MedicamentModel;
use App\Models\BandeModel;
use CodeIgniter\API\ResponseTrait;

class MedicamentController extends ResourceController
{
    use ResponseTrait;
    protected $format    = 'json';
    private $medicamentModel;
    private $distributionModel;
    private $bande;
    // Charger le service de validation
    private $validation;

    public function __construct()
    {
        $this->medicamentModel = new MedicamentModel();
        $this->distributionModel = new DistributionMedicamentModel();
        $this->bande = new BandeModel();
        // Charger le service de validation
        $this->validation = \Config\Services::validation();
    }

    /**
     * Return an array of resource objects, themselves in array format.
     *
     * @return ResponseInterface
     */
    public function index()
    {
        //
    }

    /**
     * Return the properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function show($id = null)
    {
        //
    }

    /**
     * Return a new resource object, with default properties.
     *
     * @return ResponseInterface
     */
    public function new()
    {
        //
    }

    /**
     * Create a new resource object, from "posted" parameters.
     *
     * @return ResponseInterface
     */
    public function create() {
        $data = $this->request->getJSON(true);
        // Définir les règles de validation
        $rules = [
            'ferme_id'      => 'required|integer',
            'libelle'        => 'required|min_length[2]|max_length[100]',
            'prix_unitaire'    => 'required|decimal',
            'prix_total'       => 'required|decimal',
            'quantite'    => 'required|decimal',
            'typemedicament'    => 'required|min_length[2]|max_length[100]'
        ];
        $messages = [
            'ferme_id'      => ['required' => 'La bande est requise.'],
            'libelle'        => ['required' => 'Le nom de l\'aliment est requis.',
                                  'min_length' => 'Le nom de l\'aliment doit comporter au moins 2 caractères.',
                                  'max_length' => 'Le nom de l\'aliment ne doit pas dépasser 100 caractères.',
                                  'is_unique' => 'Un Enregistrement avec ce nom existe déjà.'],
            'prix_unitaire'   => ['required' => 'Le prix unitaire est requis.'],
            'typemedicament'    => ['required' => 'Veillez choisir un type d\'aliment.'],
            'quantite'   => ['required' => 'La quantite est requis.'],
            'prix_total'   => ['required' => 'Le Total est requis.'],
        ];
        // Vérifier si les données sont valides
        if (!$this->validate($rules, $messages)) {
            // Si les données ne sont pas valides, retourner une réponse d'erreur
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de validation',
                'message' => $this->validator->getErrors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Si les données sont valides, procéder à l'enregistrement
        $datas = [
            'ferme_id'         => $data['ferme_id'],
            'nom'=> $data['libelle'],
            'prix_unitaire'      => $data['prix_unitaire'],
            'quantite'      => $data['quantite'],
            'total'      => $data['prix_total'],
            'type'      => $data['typemedicament'],
        ];
        
        //recuperer les infos sur la bande active
        $bande = $this->bande->getActiveBandeByFerme($data['ferme_id']);
        if($bande){
            $datas['id_bande'] = $bande['id'];
        }

        // Enregistrer les données dans le modèle
        $medicament = $this->medicamentModel->insert($datas);
        // Vérifier si l'enregistrement a réussi
        if (!$medicament) {
            // Si l'enregistrement a échoué, retourner une réponse d'erreur
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de l\'enregistrement du medicament',
                'message' => $this->medicamentModel->errors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Si l'enregistrement a réussi, retourner une réponse de succès
        $datas['id'] = $medicament;
        $response = [
            'status' => 201,
            'error' => false,
            'messages' => 'Enregistrement du medicament effectué avec succès',
            'data' => $datas
        ];
        return $this->respondCreated($response);
    }

    /**
     * Return the editable properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function edit($id = null)
    {
        //
    }

    /**
     * Add or update a model resource, from "posted" properties.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function update($id = null)
    {
        //
    }

    /**
     * Delete the designated resource object from the model.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function delete($id = null)
    {
        //
    }

    function listesMedicaments($iduser = null){
        if (is_null($iduser)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'ID de l\'utilisateur requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si la ferme existe
        $medicament = $this->medicamentModel->getMedicamentActiveByUserID($iduser,'active');

        if (!$medicament) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucun Médicament trouvé en stock pour cette ferme',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        foreach ($medicament as &$medocs) {
            $medocs['servie'] = (int)$this->distributionModel->getSumMedicamentDistribuer($medocs['id']);
            // Calculer la quantité restante
            $medocs['reste'] = (int)$medocs['quantite']-(int)$medocs['servie'];
        }

        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Médicaments chargés avec succès',
            'message' => null,
            'data' => $medicament
        ];
        return $this->respondCreated($response);
    }
    
    function listesAllMedicaments($iduser = null){
        if (is_null($iduser)) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'ID de l\'utilisateur requis',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Vérifier si la ferme existe
        $medicament = $this->medicamentModel->getMedicamentByUserID($iduser);

        if (!$medicament) {
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Aucun Médicament trouvé en stock pour cette ferme',
                'message' => null,
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        foreach ($medicament as &$medocs) {
            $medocs['servie'] = (int)$this->distributionModel->getSumMedicamentDistribuer($medocs['id']);
            // Calculer la quantité restante
            $medocs['reste'] = (int)$medocs['quantite']-(int)$medocs['servie'];
        }

        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Médicaments chargés avec succès',
            'message' => null,
            'data' => $medicament
        ];
        return $this->respondCreated($response);
    }
}
