<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\RESTful\ResourceController;
use App\Models\UtilisateurModel;
use App\Models\FermeModel;
use CodeIgniter\API\ResponseTrait;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;

class UtilisateurController extends ResourceController
{
    use ResponseTrait;
    protected $format    = 'json';
    private $userModel;
    private $fermeModel;
    private $validation;

    public function __construct()
    {
        $this->userModel = new UtilisateurModel();
        $this->fermeModel = new FermeModel();
        $this->validation = \Config\Services::validation();
    }

    /**
     * Return an array of resource objects, themselves in array format.
     *
     * @return ResponseInterface
     */
    public function index()
    {
        return $this->respond($this->userModel->findAll());
    }

    /**
     * Return the properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function show($id = null)
    {
        $data = $this->userModel->find($id);
        if (!$data) {
            return $this->failNotFound('Utilisateur not found');
        }
        return $this->respond($data);
    }

    /**
     * Return a new resource object, with default properties.
     *
     * @return ResponseInterface
     */
    public function new()
    {
        //
    }

    /**
     * Create a new resource object, from "posted" parameters.
     *
     * @return ResponseInterface
     */
    public function create()
    {
        $data = $this->request->getJSON(true);
        if ($this->userModel->insert($data) === false) {
            return $this->failValidationErrors($this->userModel->errors());
        }
        $data['id'] = $this->userModel->getInsertID();
        return $this->respondCreated($data);
    }

    /**
     * Return the editable properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function edit($id = null)
    {
        //
    }

    /**
     * Add or update a model resource, from "posted" properties.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function update($id = null)
    {
        $data = $this->request->getJSON(true);
        if (!$this->userModel->find($id)) {
            return $this->failNotFound('Utilisateur not found');
        }
        if ($this->userModel->update($id, $data) === false) {
            return $this->failValidationErrors($this->userModel->errors());
        }
        return $this->respond($this->userModel->find($id));
    }

    /**
     * Delete the designated resource object from the model.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function delete($id = null)
    {
        if (!$this->userModel->find($id)) {
            return $this->failNotFound('Utilisateur not found');
        }
        $this->userModel->delete($id);
        return $this->respondDeleted(['id' => $id, 'message' => 'Utilisateur deleted']);
    }

    // --- JWT Authentication ---

    public function login()
    {
        $data = $this->request->getJSON(true);
        
        // Définir les règles de validation
        $rules = [
            'email'        => 'required|valid_email',
            'mot_de_passe' => 'required|min_length[6]'
        ];
        
        $messages = [            
            "mot_de_passe" => [
                "required" => "Mot de passe requis",
                "min_length" => "Le mot de passe doit avoir au moins 6 caractères"        
            ],
            "email" => [
                "required" => "Email requis",
                "valid_email" => "L'adresse email non valide"                 
            ]
        ];
        // Valider les données
        if(!$this->validate($rules,$messages)){ 
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de Connexion! Vérifier votre saisie',
                'message' => $this->validator->getErrors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        else{
            $email = $data['email'] ?? '';
            $password = $data['mot_de_passe'] ?? '';
            $user = $this->userModel->where('email', $email)->first();

            if (!$user || !password_verify($password, $user['password'])) {
                $response = [
                    'status' => 200,
                    'error' => true,
                    'messages' => 'Erreur de Connexion! Mauvais login ou mot de passe',
                    'message' => 'Invalid credentials',
                    'data' => []
                ];
                return $this->respondCreated($response);
                //return $this->failUnauthorized('Invalid credentials');
            }

            $payload = [
                'iss' => $this->request->getServer('SERVER_ADDR'), // Issuer
                'sub' => $user['id'], // Subject
                'aud' => $this->request->getIPAddress(),  // Audience
                'iat' => time(),
                'exp' => time() + 36000, // 6 hour expiration
                'uid' => $user['id'],
                'email' => $user['email'],
                'role' => $user['role']
            ];

            $key = getenv('JWT_SECRET');

            $jwt = JWT::encode($payload, $key, 'HS256');

            unset($user['password']); // Ne jamais retourner le mot de passe hashé
            
            $response = [
                    'status' => 200,
                    'error' => false,
                    'messages' => 'Connexion Réussit!',
                    'message' => '',
                    'data' => ['token' => $jwt, 'user' => $user]
            ];
            return $this->respondCreated($response);
            //return $this->respond(['token' => $jwt, 'user' => $user]);
        }
    }

    public function logout()
    {
        // With JWT, logout is handled client-side (just delete the token).
        // Optionally, you can implement a blacklist on server side.
        return $this->respond(['message' => 'Logout successful. Please delete your token on client side.']);
    }

    public function register()
    {
        $data = $this->request->getJSON(true);

        // Définir les règles de validation
        $rules = [
            'nom'          => 'required|min_length[3]|max_length[100]',
            'email'        => 'required|valid_email|is_unique[utilisateur.email]',
            'mot_de_passe' => 'required|min_length[6]',
            'confirm_mot_de_passe' => 'required|min_length[6]|matches[mot_de_passe]',
            'role'         => 'permit_empty|in_list[admin,utilisateur]',
            'nomFerme'     => 'required|min_length[5]|is_unique[ferme.nom]',
            'localisationFerme' => 'required|min_length[5]'
        ];
        $messages = [
            "nom" => [
                "required" => "Nom requis",
                "min_length" => "Le nom doit avoir au moins 3 caractères" ,
                "max_length" => "Le nom doit avoir au plus 100 caractères"     
            ],
            "role" => [
                "required" => "role requis",   
                "in_list" => "Le role doit être admin ou utilisateur"
            ],
            "mot_de_passe" => [
                "required" => "Mot de passe requis",
                "min_length" => "Le mot de passe doit avoir au moins 6 caractères"        
            ],
            "email" => [
                "required" => "Email requis",
                "is_unique" => "Cet email existe déjà" ,
                "valid_email" => "L'adresse email non valide"                 
            ],
            "nomFerme" => [
                "required" => "Nom de la ferme requis",
                "min_length" => "Le nom de la ferme doit avoir au moins 5 caractères",
                "is_unique" => "Ce nom de ferme existe déjà"
            ],
            "localisationFerme" => [
                "required" => "Localisation de la ferme requise",
                "min_length" => "La localisation de la ferme doit avoir au moins 5 caractères"
            ],
            "confirm_mot_de_passe" => [
                "required" => "Confirmation du mot de passe requise",
                "min_length" => "La confirmation du mot de passe doit avoir au moins 6 caractères",
                "matches" => "Les mots de passe ne correspondent pas"
            ]
        ];

        if(!$this->validate($rules,$messages)){ 
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Compte Non créé ! Vérifier votre saisie',
                'message' => $this->validator->getErrors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }

        // Hash du mot de passe
        $datas = [
            'nom'          => $data['nom'],
            'email'        => $data['email'],
            'password' => password_hash($data['mot_de_passe'], PASSWORD_DEFAULT),
            'role'         => 'utilisateur',
            'statut'         => 'active',
            'etat'         => 1,
            'ipAddress'   => $this->request->getIPAddress(),
            'last_connect' => date('Y-m-d H:i:s'),
        ];

        if ($this->userModel->insert($datas) === false) {
            $response = [
                'status' => 201,
                'error' => true,
                'messages' => 'Compte Non créé ! Vérifier votre saisie',
                'message' => $this->userModel->errors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        $data['id'] = $this->userModel->getInsertID();
        //unset($data['mot_de_passe']); // Ne jamais retourner le mot de passe hashé
        //creer la ferme
        $datas = [
            'nom'         => $data['nomFerme'],
            'localisation'=> $data['localisationFerme'],
            'utilisateur_id' => $data['id'], // Assurez-vous que l'ID de l'utilisateur est fourni
            'statut'      => 'active', // Valeur par défaut si non fournie 
            // Ajoute d'autres champs si besoin
        ];
        $this->fermeModel->insert($datas);

        $response = [
                    'status' => 200,
                    'error' => false,
                    'messages' => 'Compte créé avec succès ! Vous pouvez vous connecter.',
                    'message' => '',
                    'data' => $data
                ];
        return $this->respondCreated($response);
    }
}
