<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\RESTful\ResourceController;
use App\Models\VenteModel;
use CodeIgniter\API\ResponseTrait;
use App\Models\CoutProductionBandeModel;

class VenteController extends ResourceController
{
    use ResponseTrait;
    protected $format    = 'json';
    private $venteModel;
    private $validation;
    private $coutModel;

    public function __construct()
    {
        $this->venteModel = new VenteModel();
        $this->validation = \Config\Services::validation();
        $this->coutModel = new CoutProductionBandeModel();
    }

    /**
     * Return an array of resource objects, themselves in array format.
     *
     * @return ResponseInterface
     */
    public function index() {
        $vente = $this->venteModel->loadVente('active');
        foreach ($vente as &$v) {
            //calculer  le benefice unitaire et total
            $v['benefice_unitaire'] = round($v['prix_vente'] - $v['prix_unitaire'],2);
            $v['total_benefice'] = round($v['montant_total'] - $v['total_vente'],2);
        }

        $response = [
            'status' => 200,
            'error' => false,
            'messages' => 'Ventes chargées avec succès',
            'message' => null,
            'data' => $vente
        ];
        return $this->respondCreated($response);
    }

    /**
     * Return the properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function show($id = null)
    {
        $data = $this->model->find($id);
        if (!$data) {
            return $this->failNotFound('Vente not found');
        }
        return $this->respond($data);
    }

    /**
     * Return a new resource object, with default properties.
     *
     * @return ResponseInterface
     */
    public function new()
    {
        //
    }

    /**
     * Create a new resource object, from "posted" parameters.
     *
     * @return ResponseInterface
     */
    public function create() {
       
        $data = $this->request->getJSON(true);
        // Définir les règles de validation
        $rules = [
            'ferme_id'      => 'required|integer',
            'bande_id'      => 'required|integer',
            'prixUnitaire'    => 'required|decimal',
            'total'       => 'required|decimal',
            'nombre'    => 'required|decimal',
            'nombreRestant' => 'required|decimal',
            'datevente' => 'required|valid_date[Y-m-d]', // Assurez-vous que la date est au format Y-m-d
        ];
        $messages = [
            'ferme_id'      => ['required' => 'La ferme est requise.'],
            'bande_id'      => ['required' => 'La bande est requise.'],
            'prixUnitaire'    => ['required' => 'Le prix unitaire est requis.'],
            'total'       => ['required' => 'Le total est requis.'],
            'nombre'    => ['required' => 'Le nombre de sujets vendus est requis.'],
            'nombreRestant' => ['required' => 'Le nombre de sujets restant est requis.'], 
            'datevente' => ['required' => 'La date de vente est requise.', 'valid_date' => 'La date de vente doit être au format Y-m-d.']
        ];
        if (!$this->validate($rules, $messages)) {
             // Si les données ne sont pas valides, retourner une réponse d'erreur
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur de validation',
                'message' => $this->validator->getErrors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // recuperer le cout de production pour la bande a la date de vente
        $coutRow = $this->coutModel->getCostByDate($data['bande_id'], $data['datevente']);
        if (!$coutRow) {
            $coutRow = $this->coutModel->getLatestCostBefore($data['bande_id'], $data['datevente']);
        }
        $cout_production = $coutRow ? (float)$coutRow['cout_production'] : 0.00;
         // Si les données sont valides, procéder à l'enregistrement
        $datas = [
            'bande_id'         => $data['bande_id'],
            'prix_vente'      => $data['prixUnitaire'],
            'quantite'      => $data['nombre'],
            'montant_total'      => $data['total'],
            'date' => $data['datevente'],
            'ferme_id' => $data['ferme_id'],
            'prix_unitaire' => $cout_production,
            'total_vente' => $cout_production * $data['nombre']
        ];
        // Enregistrer les données dans le modèle
        $vente = $this->venteModel->insert($datas);
        // Vérifier si l'enregistrement a réussi
        if (!$vente) {
            // Si l'enregistrement a échoué, retourner une réponse d'erreur
            $response = [
                'status' => 200,
                'error' => true,
                'messages' => 'Erreur lors de l\'enregistrement de la vente',
                'message' => $this->venteModel->errors(),
                'data' => []
            ];
            return $this->respondCreated($response);
        }
        // Si l'enregistrement a réussi, retourner une réponse de succès
        $datas['id'] = $vente;
        $response = [
            'status' => 201,
            'error' => false,
            'messages' => 'Enregistrement de la vente effectué avec succès',
            'data' => $datas
        ];
        return $this->respondCreated($response);
    }   

    /**
     * Return the editable properties of a resource object.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function edit($id = null)
    {
        //
    }

    /**
     * Add or update a model resource, from "posted" properties.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function update($id = null){
        $data = $this->request->getJSON(true);
        if (!$this->model->find($id)) {
            return $this->failNotFound('Vente not found');
        }
        if ($this->model->update($id, $data) === false) {
            return $this->failValidationErrors($this->model->errors());
        }
        return $this->respond($this->model->find($id));
    }

    /**
     * Delete the designated resource object from the model.
     *
     * @param int|string|null $id
     *
     * @return ResponseInterface
     */
    public function delete($id = null)
    {
        if (!$this->model->find($id)) {
            return $this->failNotFound('Vente not found');
        }
        $this->model->delete($id);
        return $this->respondDeleted(['id' => $id, 'message' => 'Vente deleted']);
    }
}
