<?php

namespace App\Models;

use CodeIgniter\Model;

class VenteModel extends Model
{
    protected $table            = 'vente';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = false;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'bande_id',
        'quantite',
        'prix_vente',
        'montant_total',
        'date',
        'statut',
        'created_at',
        'updated_at',
        'deleted_at',
        'ferme_id',
        'prix_unitaire',
        'total_vente'
    ];

    protected bool $allowEmptyInserts = false;
    protected bool $updateOnlyChanged = true;

    protected array $casts = [];
    protected array $castHandlers = [];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules      = [];
    protected $validationMessages   = [];
    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert   = [];
    protected $afterInsert    = ['afterInsertCreateBenefice'];
    protected $beforeUpdate   = [];
    protected $afterUpdate    = [];
    protected $beforeFind     = [];
    protected $afterFind      = [];
    protected $beforeDelete   = [];
    protected $afterDelete    = [];

    function getQuantiteVente($bande_id) {   
        $res = $this->select('SUM(quantite) as quantite')
                    ->where('bande_id', $bande_id)
                    ->first();
        return $res ? (int)$res['quantite'] : 0;
    }

    function getTotalVente($bande_id) {   
        $res = $this->select('SUM(total_vente) as total')
                    ->where('bande_id', $bande_id)
                    ->first();
        return $res ? (int)$res['total'] : 0;
    }

    function loadVente($etat){
        return $this->select('vente.*, ferme.nom as ferme_nom, localisation, libelle, entreprise, souche')
                    ->join('bande', 'bande.id = vente.bande_id', 'left')
                    ->join('ferme', 'ferme.id = bande.ferme_id', 'left')
                    ->where('bande.statut', $etat)
                    ->orderBy('vente.id', 'DESC')
                    ->findAll();
    }

    /**
     * Callback executed after a Vente is inserted.
     * It creates a BeneficeVenteBande entry using the CoutProductionBande for the vente date
     */
    protected function afterInsertCreateBenefice(array $data) {
        $insertedId = null;
        if (isset($data['id'])) {
            $insertedId = is_array($data['id']) ? ($data['id'][0] ?? null) : $data['id'];
        }
        if (!$insertedId) {
            return;
        }

        $vente = $this->find($insertedId);
        if (!$vente) {
            return;
        }

        $bandeId = $vente['bande_id'] ?? null;
        $date = isset($vente['date']) ? date('Y-m-d', strtotime($vente['date'])) : null;
        //
        $montant_total = isset($vente['montant_total']) ? (float)$vente['montant_total'] : 0.00;
        $quantite = isset($vente['quantite']) ? (int)$vente['quantite'] : 1;
        $ferme_id = $vente['ferme_id'] ?? null;
        $prix_vente = isset($vente['prix_vente']) ? (float)$vente['prix_vente'] : 0.00;
        $prix_unitaire = isset($vente['prix_unitaire']) ? (float)$vente['prix_unitaire'] : 0.00;
        $total_vente = isset($vente['total_vente']) ? (float)$vente['total_vente'] : 0.00;

        if (!$bandeId || !$date) {
            return;
        }


        $beneficeModel = new \App\Models\BeneficeVenteBandeModel();
        if ($beneficeModel->where('vente_id', $insertedId)->first()) {
            return;
        }

        $beneficeModel->insert([
            'vente_id' => $insertedId,
            'bande_id' => $bandeId,
            'ferme_id' => $ferme_id,
            'date_vente' => $date,
            'montant_vente' => $montant_total,
            'prix_unitaire' => $prix_vente,
            'benefice_unitaire' => $prix_vente - $prix_unitaire,
            'quantite' => $quantite,
            'cout_production' => $prix_unitaire, // per unit
            'benefice' => ($prix_vente - $prix_unitaire)*$quantite,
        ]);
    } 
}